import re
import string
from collections import Counter

def get_unique_words_basic(text):
    """
    方法1: 基础方法 - 使用正则表达式和集合
    """
    # 转换为小写
    text = text.lower()
    
    # 使用正则表达式提取单词（只保留字母和数字）
    words = re.findall(r'\b[a-zA-Z0-9]+\b', text)
    
    # 去重并排序
    unique_words = sorted(set(words))
    
    return unique_words

def get_unique_words_advanced(text):
    """
    方法2: 高级方法 - 更精细的文本清理
    """
    # 转换为小写
    text = text.lower()
    
    # 移除标点符号
    translator = str.maketrans('', '', string.punctuation)
    text = text.translate(translator)
    
    # 分割单词
    words = text.split()
    
    # 过滤空字符串和纯数字（可选）
    words = [word for word in words if word and not word.isdigit()]
    
    # 去重并排序
    unique_words = sorted(set(words))
    
    return unique_words

def get_words_with_frequency(text):
    """
    方法3: 获取单词及其出现频率
    """
    text = text.lower()
    words = re.findall(r'\b[a-zA-Z0-9]+\b', text)
    
    # 计算词频
    word_count = Counter(words)
    
    # 按频率排序
    sorted_by_frequency = word_count.most_common()
    
    # 获取唯一单词（按字母顺序）
    unique_words = sorted(word_count.keys())
    
    return unique_words, sorted_by_frequency

def analyze_text_comprehensive(text):
    """
    方法4: 综合分析
    """
    print("=== 文本分析 ===")
    
    # 基本统计
    total_chars = len(text)
    total_words = len(text.split())
    
    # 获取唯一单词
    unique_words = get_unique_words_basic(text)
    unique_count = len(unique_words)
    
    # 获取词频
    _, word_frequency = get_words_with_frequency(text)
    
    # 输出统计信息
    print(f"总字符数: {total_chars}")
    print(f"总单词数: {total_words}")
    print(f"唯一单词数: {unique_count}")
    print(f"词汇丰富度: {unique_count/total_words:.2%}")
    
    print(f"\n前10个最常用单词:")
    for word, count in word_frequency[:10]:
        print(f"  {word}: {count}次")
    
    print(f"\n前20个唯一单词（字母顺序）:")
    for i, word in enumerate(unique_words[:20], 1):
        print(f"  {i:2d}. {word}")
    
    return unique_words

def save_words_to_file(words, filename="unique_words.txt"):
    """
    方法5: 保存单词列表到文件
    """
    with open(filename, 'w', encoding='utf-8') as f:
        for word in words:
            f.write(word + '\n')
    
    print(f"单词列表已保存到 {filename}")

# 示例文本（你的文档内容）

if __name__ == "__main__":
    with open('reasoning.txt', 'r', encoding='utf-8') as file:
        sample_text = file.read()

        # 测试不同方法
        print("方法1 - 基础方法:")
        words1 = get_unique_words_basic(sample_text)
        print(f"找到 {len(words1)} 个唯一单词")
        print("前10个:", words1[:10])
        
        print("\n方法2 - 高级方法:")
        words2 = get_unique_words_advanced(sample_text)
        print(f"找到 {len(words2)} 个唯一单词")
        print("前10个:", words2[:10])
        
        print("\n方法3 - 带词频统计:")
        unique_words, frequency = get_words_with_frequency(sample_text)
        print(f"找到 {len(unique_words)} 个唯一单词")
        print("词频最高的5个:", frequency[:5])
        
        print("\n方法4 - 综合分析:")
        all_words = analyze_text_comprehensive(sample_text)
        
        # 保存结果
        print(f"\n是否保存到文件? (输入y确认)")
        save_words_to_file(all_words)  # 取消注释以保存文件

# 处理你的完整文档
def process_your_document():
    """
    处理你提供的完整文档
    """
    # 这里放入你的完整文档文本
    your_text = """[在这里粘贴你的完整文档内容]"""
    
    print("处理你的文档...")
    unique_words = get_unique_words_basic(your_text)
    
    print(f"你的文档包含 {len(unique_words)} 个唯一单词")
    
    # 输出所有唯一单词
    print("\n所有唯一单词:")
    for i, word in enumerate(unique_words, 1):
        print(f"{i:3d}. {word}")
    
    return unique_words

# 使用示例：
# your_unique_words = process_your_document()